<?php
// get_booking_details.php
session_start();
require_once 'config.php';

if (!isAdminLoggedIn()) {
    http_response_code(403);
    exit('Access denied');
}

$booking_id = intval($_GET['id'] ?? 0);

// ── Main booking row ──────────────────────────────────────────────
$stmt = $pdo->prepare("SELECT * FROM bookings WHERE id = ?");
$stmt->execute([$booking_id]);
$booking = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$booking) {
    echo '<div style="text-align:center;padding:40px;color:#b52b2b;">
            <i class="fas fa-exclamation-circle fa-2x"></i>
            <p style="margin-top:12px;font-weight:600;">Booking not found</p>
          </div>';
    exit;
}

// ── Rooms (separate query — avoids GROUP_CONCAT NULL issue) ───────
$rooms_stmt = $pdo->prepare("SELECT room_name FROM booking_rooms WHERE booking_id = ?");
$rooms_stmt->execute([$booking_id]);
$rooms = $rooms_stmt->fetchAll(PDO::FETCH_COLUMN);

// ── Event nature (separate query) ─────────────────────────────────
$nature_stmt = $pdo->prepare("SELECT nature_type FROM booking_nature WHERE booking_id = ?");
$nature_stmt->execute([$booking_id]);
$natures = $nature_stmt->fetchAll(PDO::FETCH_COLUMN);

// ── Helper: safe string output ────────────────────────────────────
function safe(mixed $val, string $fallback = '—'): string {
    $v = trim((string)($val ?? ''));
    return $v !== '' ? htmlspecialchars($v) : $fallback;
}

// ── Status badge helper ───────────────────────────────────────────
$status_map = [
    'pending'  => ['bg' => '#fef9ec', 'color' => '#92650a', 'dot' => '#b8973b'],
    'approved' => ['bg' => '#eaf5ee', 'color' => '#1e6b43', 'dot' => '#1e6b43'],
    'rejected' => ['bg' => '#fdf0f0', 'color' => '#b52b2b', 'dot' => '#b52b2b'],
];
$s  = $booking['status'] ?? 'pending';
$sc = $status_map[$s] ?? $status_map['pending'];
?>

<style>
    /* ── Scoped modal detail styles ─────────────────────────────── */
    .bd-wrap {
        font-family: 'DM Sans', sans-serif;
        font-size: 0.84rem;
        color: #0d1b2e;
    }

    /* Section header inside modal */
    .bd-section {
        margin-bottom: 22px;
    }
    .bd-section-title {
        font-size: 0.62rem;
        font-weight: 700;
        letter-spacing: 0.12em;
        text-transform: uppercase;
        color: #7a8799;
        margin-bottom: 10px;
        padding-bottom: 6px;
        border-bottom: 1px solid #e2ddd6;
        display: flex;
        align-items: center;
        gap: 7px;
    }
    .bd-section-title i { color: #b8973b; font-size: 0.7rem; }

    /* Two-column grid of label/value pairs */
    .bd-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 14px 24px;
    }
    .bd-grid.full { grid-template-columns: 1fr; }

    .bd-item {}
    .bd-item.span2 { grid-column: 1 / -1; }

    .bd-label {
        font-size: 0.65rem;
        font-weight: 700;
        letter-spacing: 0.08em;
        text-transform: uppercase;
        color: #7a8799;
        margin-bottom: 3px;
    }
    .bd-val {
        font-size: 0.84rem;
        color: #0d1b2e;
        font-weight: 500;
        line-height: 1.5;
    }
    .bd-val.muted { color: #3a4a62; font-weight: 400; }
    .bd-val.mono  { font-family: monospace; font-size: 0.82rem; }

    /* Inline badge */
    .bd-badge {
        display: inline-flex;
        align-items: center;
        gap: 5px;
        padding: 4px 10px;
        border-radius: 100px;
        font-size: 0.68rem;
        font-weight: 700;
        letter-spacing: 0.05em;
        text-transform: uppercase;
    }
    .bd-badge::before {
        content: '';
        width: 5px; height: 5px;
        border-radius: 50%;
        flex-shrink: 0;
    }

    /* Pills for lists */
    .bd-pills { display: flex; flex-wrap: wrap; gap: 6px; }
    .bd-pill {
        display: inline-block;
        padding: 3px 10px;
        border-radius: 100px;
        font-size: 0.72rem;
        font-weight: 600;
        background: #ede8dd;
        color: #3a4a62;
    }
    .bd-pill.empty {
        background: #f5f5f5;
        color: #aaa;
        font-weight: 400;
        font-style: italic;
    }

    /* Warning box for DB issues */
    .bd-warn {
        background: #fef9ec;
        border: 1px solid #f0d98a;
        border-radius: 6px;
        padding: 8px 12px;
        font-size: 0.76rem;
        color: #92650a;
        display: flex;
        align-items: flex-start;
        gap: 8px;
        margin-top: 4px;
    }
    .bd-warn i { margin-top: 1px; flex-shrink: 0; }

    /* Info banner at top */
    .bd-ref-bar {
        display: flex;
        align-items: center;
        justify-content: space-between;
        background: #0d1b2e;
        border-radius: 8px;
        padding: 14px 18px;
        margin-bottom: 22px;
    }
    .bd-ref-bar .ref-num {
        font-family: 'Playfair Display', serif;
        font-size: 1.1rem;
        font-weight: 400;
        color: #fff;
        display: flex; align-items: center; gap: 10px;
    }
    .bd-ref-bar .ref-num i { color: #b8973b; font-size: 0.85rem; }
    .bd-ref-bar .ref-date {
        font-size: 0.72rem;
        color: rgba(255,255,255,0.4);
    }
</style>

<div class="bd-wrap">

    <!-- Reference bar -->
    <div class="bd-ref-bar">
        <div class="ref-num">
            <i class="fas fa-hashtag"></i>
            Booking #<?= str_pad($booking['id'], 4, '0', STR_PAD_LEFT) ?>
        </div>
        <div>
            <span class="bd-badge" style="background:<?= $sc['bg'] ?>;color:<?= $sc['color'] ?>;">
                <span style="width:5px;height:5px;border-radius:50%;background:<?= $sc['dot'] ?>;display:inline-block;"></span>
                <?= ucfirst($s) ?>
            </span>
        </div>
    </div>

    <!-- 1. Organisation & Contact -->
    <div class="bd-section">
        <div class="bd-section-title"><i class="fas fa-building"></i> Organisation &amp; Contact</div>
        <div class="bd-grid">
            <div class="bd-item">
                <div class="bd-label">Organisation</div>
                <div class="bd-val"><?= safe($booking['organization_name']) ?></div>
            </div>
            <div class="bd-item">
                <div class="bd-label">Representative</div>
                <div class="bd-val"><?= safe($booking['representative_name']) ?></div>
            </div>
            <div class="bd-item">
                <div class="bd-label">Phone</div>
                <div class="bd-val mono"><?= safe($booking['phone']) ?></div>
            </div>
            <div class="bd-item">
                <div class="bd-label">Email</div>
                <div class="bd-val mono"><?= safe($booking['email']) ?></div>
            </div>
        </div>
    </div>

    <!-- 2. Event Information -->
    <div class="bd-section">
        <div class="bd-section-title"><i class="fas fa-calendar-alt"></i> Event Information</div>
        <div class="bd-grid">
            <div class="bd-item span2">
                <div class="bd-label">Event Name</div>
                <div class="bd-val"><?= safe($booking['event_name']) ?></div>
            </div>
            <div class="bd-item span2">
                <div class="bd-label">Description</div>
                <div class="bd-val muted"><?= nl2br(safe($booking['event_description'], 'No description provided.')) ?></div>
            </div>
            <div class="bd-item">
                <div class="bd-label">Event Date(s)</div>
                <div class="bd-val"><?= safe($booking['event_dates']) ?></div>
            </div>
            <div class="bd-item">
                <div class="bd-label">Time</div>
                <div class="bd-val mono">
                    <?= safe($booking['start_time']) ?> — <?= safe($booking['end_time']) ?>
                </div>
            </div>
            <div class="bd-item">
                <div class="bd-label">Max Participants</div>
                <div class="bd-val"><?= safe($booking['max_participants']) ?></div>
            </div>
            <div class="bd-item">
                <div class="bd-label">Elaborate Set-up</div>
                <div class="bd-val"><?= ucfirst(safe($booking['elaborate_setup'] ?? '', 'Not specified')) ?></div>
            </div>
        </div>
    </div>

    <!-- 3. Rooms -->
    <div class="bd-section">
        <div class="bd-section-title"><i class="fas fa-door-open"></i> Rooms Requested</div>
        <?php if (!empty($rooms)): ?>
            <div class="bd-pills">
                <?php foreach ($rooms as $room): ?>
                    <span class="bd-pill"><?= htmlspecialchars($room) ?></span>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="bd-pills">
                <span class="bd-pill empty">No room data found</span>
            </div>
            <div class="bd-warn">
                <i class="fas fa-info-circle"></i>
                <span>Room selections were not saved to the <code>booking_rooms</code> table for this submission. Check that <code>submit_booking.php</code> is inserting room rows correctly.</span>
            </div>
        <?php endif; ?>
    </div>

    <!-- 4. Event Nature -->
    <div class="bd-section">
        <div class="bd-section-title"><i class="fas fa-tag"></i> Nature of Event</div>
        <?php if (!empty($natures)): ?>
            <div class="bd-pills">
                <?php foreach ($natures as $n): ?>
                    <span class="bd-pill"><?= ucfirst(htmlspecialchars($n)) ?></span>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="bd-pills">
                <span class="bd-pill empty">No nature data found</span>
            </div>
            <div class="bd-warn">
                <i class="fas fa-info-circle"></i>
                <span>Event nature selections were not saved to the <code>booking_nature</code> table. Check that <code>submit_booking.php</code> inserts checkbox values.</span>
            </div>
        <?php endif; ?>
    </div>

    <!-- 5. Special Requirements -->
    <?php
    $hasMusic = !empty($booking['music_event']) && $booking['music_event'] === 'yes';
    $hasVVIP  = !empty($booking['vvip'])        && $booking['vvip']        === 'yes';
    $hasOther = !empty(trim($booking['other_info'] ?? ''));
    if ($hasMusic || $hasVVIP || $hasOther):
    ?>
    <div class="bd-section">
        <div class="bd-section-title"><i class="fas fa-star"></i> Special Requirements</div>
        <div class="bd-grid">
            <?php if ($hasMusic): ?>
            <div class="bd-item span2">
                <div class="bd-label">Music / Singing Arrangements</div>
                <div class="bd-val muted"><?= nl2br(safe($booking['music_details'])) ?></div>
            </div>
            <?php endif; ?>
            <?php if ($hasVVIP): ?>
            <div class="bd-item span2">
                <div class="bd-label">VVIP Guests</div>
                <div class="bd-val muted"><?= nl2br(safe($booking['vvip_names'])) ?></div>
            </div>
            <?php endif; ?>
            <?php if ($hasOther): ?>
            <div class="bd-item span2">
                <div class="bd-label">Additional Information</div>
                <div class="bd-val muted"><?= nl2br(safe($booking['other_info'])) ?></div>
            </div>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>

    <!-- 6. Declaration & Submission -->
    <div class="bd-section" style="margin-bottom:0;">
        <div class="bd-section-title"><i class="fas fa-file-signature"></i> Declaration &amp; Submission</div>
        <div class="bd-grid">
            <div class="bd-item">
                <div class="bd-label">Declared By</div>
                <div class="bd-val"><?= safe($booking['declaration_name']) ?></div>
            </div>
            <div class="bd-item">
                <div class="bd-label">Submission Date</div>
                <div class="bd-val"><?= date('d M Y, g:i A', strtotime($booking['submission_date'])) ?></div>
            </div>
        </div>
    </div>

</div>