<?php
// admin_dashboard.php
session_start();
require_once 'config.php';

if (!isAdminLoggedIn()) {
    header('Location: admin_login.php');
    exit;
}

$status_filter = $_GET['status'] ?? 'all';
$search_term   = $_GET['search'] ?? '';
$date_from     = $_GET['date_from'] ?? '';
$date_to       = $_GET['date_to'] ?? '';

$query = "
    SELECT b.*,
           GROUP_CONCAT(DISTINCT br.room_name) as rooms,
           GROUP_CONCAT(DISTINCT bn.nature_type) as event_nature
    FROM bookings b
    LEFT JOIN booking_rooms br ON b.id = br.booking_id
    LEFT JOIN booking_nature bn ON b.id = bn.booking_id
    WHERE 1=1
";
$params = [];

if ($status_filter !== 'all') {
    $query .= " AND b.status = ?";
    $params[] = $status_filter;
}
if (!empty($search_term)) {
    $query .= " AND (b.organization_name LIKE ? OR b.event_name LIKE ? OR b.representative_name LIKE ?)";
    $like = "%$search_term%";
    $params[] = $like; $params[] = $like; $params[] = $like;
}
if (!empty($date_from)) { $query .= " AND DATE(b.submission_date) >= ?"; $params[] = $date_from; }
if (!empty($date_to))   { $query .= " AND DATE(b.submission_date) <= ?"; $params[] = $date_to; }
$query .= " GROUP BY b.id ORDER BY b.submission_date DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$bookings = $stmt->fetchAll();

$stats_stmt = $pdo->query("
    SELECT COUNT(*) as total,
           SUM(CASE WHEN status='pending'  THEN 1 ELSE 0 END) as pending,
           SUM(CASE WHEN status='approved' THEN 1 ELSE 0 END) as approved,
           SUM(CASE WHEN status='rejected' THEN 1 ELSE 0 END) as rejected
    FROM bookings
");
$stats = $stats_stmt->fetch();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard — CEDI Conference Centre</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Playfair+Display:ital,wght@0,400;0,500;1,400&family=DM+Sans:wght@300;400;500;600&display=swap">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --navy:        #0d1b2e;
            --navy-mid:    #162338;
            --navy-light:  #1d3050;
            --navy-faint:  #243660;
            --gold:        #b8973b;
            --gold-light:  #d4b05a;
            --gold-pale:   #f5edd8;
            --cream:       #f8f5ef;
            --cream-dark:  #ede8dd;
            --white:       #ffffff;
            --text-dark:   #0d1b2e;
            --text-mid:    #3a4a62;
            --text-light:  #7a8799;
            --border:      #e2ddd6;
            --success:     #1e6b43;
            --success-bg:  #eaf5ee;
            --danger:      #b52b2b;
            --danger-bg:   #fdf0f0;
            --warning:     #92650a;
            --warning-bg:  #fef9ec;
            --info:        #1a5c8a;
            --info-bg:     #eef5fb;
            --sidebar-w:   260px;
            --radius:      8px;
        }

        *, *::before, *::after { margin: 0; padding: 0; box-sizing: border-box; }

        body {
            font-family: 'DM Sans', sans-serif;
            background: var(--cream);
            color: var(--text-dark);
            min-height: 100vh;
            display: flex;
            -webkit-font-smoothing: antialiased;
        }

        /* ═══════════════════════════
           SIDEBAR
        ═══════════════════════════ */
        .sidebar {
            width: var(--sidebar-w);
            min-height: 100vh;
            background: var(--navy);
            position: fixed;
            top: 0; left: 0;
            z-index: 200;
            display: flex;
            flex-direction: column;
            border-right: 1px solid rgba(184,151,59,0.15);
        }

        .sidebar-logo {
            padding: 28px 24px 20px;
            border-bottom: 1px solid rgba(255,255,255,0.07);
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .logo-mark {
            width: 44px; height: 44px;
            border-radius: 10px;
            background: rgba(184,151,59,0.15);
            border: 1px solid rgba(184,151,59,0.35);
            display: flex; align-items: center; justify-content: center;
            flex-shrink: 0;
        }
        .logo-mark img {
            width: 36px; height: 36px;
            object-fit: contain; border-radius: 6px;
        }
        .logo-mark i { color: var(--gold-light); font-size: 1.2rem; }

        .logo-text strong {
            display: block;
            font-family: 'Playfair Display', serif;
            font-size: 0.95rem;
            font-weight: 500;
            color: var(--white);
            line-height: 1.2;
        }
        .logo-text span {
            font-size: 0.62rem;
            color: rgba(255,255,255,0.35);
            letter-spacing: 0.07em;
            text-transform: uppercase;
        }

        .sidebar-section-label {
            font-size: 0.6rem;
            font-weight: 600;
            letter-spacing: 0.14em;
            text-transform: uppercase;
            color: rgba(255,255,255,0.25);
            padding: 20px 24px 8px;
        }

        .nav-item {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 11px 24px;
            color: rgba(255,255,255,0.5);
            text-decoration: none;
            font-size: 0.82rem;
            font-weight: 500;
            transition: all .2s;
            position: relative;
            border-radius: 0;
        }
        .nav-item i { width: 18px; font-size: 0.85rem; }
        .nav-item:hover { color: rgba(255,255,255,0.85); background: rgba(255,255,255,0.04); }
        .nav-item.active {
            color: var(--gold-light);
            background: rgba(184,151,59,0.1);
        }
        .nav-item.active::before {
            content: '';
            position: absolute;
            left: 0; top: 0; bottom: 0;
            width: 3px;
            background: var(--gold);
            border-radius: 0 2px 2px 0;
        }

        .sidebar-footer {
            margin-top: auto;
            padding: 20px 24px;
            border-top: 1px solid rgba(255,255,255,0.07);
        }
        .sidebar-footer a {
            display: flex; align-items: center; gap: 10px;
            font-size: 0.78rem; color: rgba(255,255,255,0.35);
            text-decoration: none; transition: color .2s;
        }
        .sidebar-footer a:hover { color: var(--danger); }
        .sidebar-footer a i { font-size: 0.85rem; }

        /* ═══════════════════════════
           MAIN CONTENT
        ═══════════════════════════ */
        .main {
            margin-left: var(--sidebar-w);
            flex: 1;
            display: flex;
            flex-direction: column;
            min-height: 100vh;
        }

        /* Top bar */
        .topbar {
            background: var(--white);
            border-bottom: 1px solid var(--border);
            padding: 0 36px;
            height: 60px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            position: sticky; top: 0; z-index: 100;
        }

        .topbar-left {
            display: flex;
            flex-direction: column;
        }
        .topbar-left h1 {
            font-family: 'Playfair Display', serif;
            font-size: 1.1rem;
            font-weight: 500;
            color: var(--navy);
            line-height: 1;
        }
        .topbar-left span {
            font-size: 0.7rem;
            color: var(--text-light);
            margin-top: 2px;
        }

        .topbar-right {
            display: flex;
            align-items: center;
            gap: 16px;
        }

        .topbar-btn {
            display: flex; align-items: center; gap: 7px;
            padding: 8px 16px;
            border: 1.5px solid var(--border);
            border-radius: var(--radius);
            background: transparent;
            font-family: 'DM Sans', sans-serif;
            font-size: 0.76rem; font-weight: 600;
            color: var(--text-mid);
            cursor: pointer; text-decoration: none;
            transition: all .2s;
        }
        .topbar-btn:hover { border-color: var(--navy); color: var(--navy); }
        .topbar-btn.gold { background: var(--gold); border-color: var(--gold); color: var(--navy); }
        .topbar-btn.gold:hover { background: var(--gold-light); border-color: var(--gold-light); }

        .topbar-avatar {
            width: 36px; height: 36px;
            border-radius: 50%;
            background: var(--navy);
            display: flex; align-items: center; justify-content: center;
            color: var(--gold-light);
            font-size: 0.85rem;
            font-weight: 600;
            border: 2px solid rgba(184,151,59,0.3);
        }

        /* ═══════════════════════════
           PAGE CONTENT
        ═══════════════════════════ */
        .content { padding: 32px 36px; flex: 1; }

        /* ── STAT CARDS ── */
        .stats-row {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 16px;
            margin-bottom: 28px;
        }

        .stat-card {
            background: var(--white);
            border: 1px solid var(--border);
            border-radius: var(--radius);
            padding: 22px 22px 18px;
            position: relative;
            overflow: hidden;
            transition: box-shadow .2s, transform .2s;
        }
        .stat-card:hover { transform: translateY(-2px); box-shadow: 0 8px 24px rgba(13,27,46,0.08); }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0; left: 0; right: 0;
            height: 3px;
        }
        .stat-card.s-total::before   { background: var(--navy); }
        .stat-card.s-pending::before { background: var(--gold); }
        .stat-card.s-approved::before{ background: var(--success); }
        .stat-card.s-rejected::before{ background: var(--danger); }

        .stat-top {
            display: flex; justify-content: space-between; align-items: flex-start;
            margin-bottom: 14px;
        }
        .stat-label {
            font-size: 0.68rem; font-weight: 600;
            letter-spacing: 0.1em; text-transform: uppercase;
            color: var(--text-light);
        }
        .stat-icon-wrap {
            width: 36px; height: 36px;
            border-radius: 8px;
            display: flex; align-items: center; justify-content: center;
            font-size: 0.9rem;
        }
        .s-total   .stat-icon-wrap { background: rgba(13,27,46,0.08);   color: var(--navy); }
        .s-pending .stat-icon-wrap { background: var(--warning-bg);      color: var(--gold); }
        .s-approved .stat-icon-wrap{ background: var(--success-bg);      color: var(--success); }
        .s-rejected .stat-icon-wrap{ background: var(--danger-bg);       color: var(--danger); }

        .stat-number {
            font-family: 'Playfair Display', serif;
            font-size: 2.4rem;
            font-weight: 400;
            color: var(--navy);
            line-height: 1;
            margin-bottom: 4px;
        }
        .stat-sub {
            font-size: 0.7rem;
            color: var(--text-light);
        }

        /* ── FILTER PANEL ── */
        .panel {
            background: var(--white);
            border: 1px solid var(--border);
            border-radius: var(--radius);
            margin-bottom: 20px;
            overflow: hidden;
        }
        .panel-head {
            padding: 14px 22px;
            border-bottom: 1px solid var(--border);
            display: flex; align-items: center; justify-content: space-between;
        }
        .panel-head h2 {
            font-size: 0.78rem; font-weight: 600;
            letter-spacing: 0.07em; text-transform: uppercase;
            color: var(--text-mid);
            display: flex; align-items: center; gap: 8px;
        }
        .panel-head h2 i { color: var(--gold); font-size: 0.75rem; }
        .panel-body { padding: 20px 22px; }

        .filter-row {
            display: grid;
            grid-template-columns: 160px 1fr 160px 160px auto;
            gap: 14px;
            align-items: end;
        }
        .fg { display: flex; flex-direction: column; gap: 5px; }
        .fg label {
            font-size: 0.68rem; font-weight: 600;
            letter-spacing: 0.07em; text-transform: uppercase;
            color: var(--text-light);
        }
        .fg input, .fg select {
            padding: 9px 12px;
            border: 1.5px solid var(--border);
            border-radius: var(--radius);
            font-family: 'DM Sans', sans-serif;
            font-size: 0.82rem; color: var(--text-dark);
            background: var(--white); transition: border-color .2s;
            -webkit-appearance: none;
        }
        .fg select {
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='10' height='6'%3E%3Cpath d='M0 0l5 6 5-6z' fill='%237a8799'/%3E%3C/svg%3E");
            background-repeat: no-repeat; background-position: right 10px center; cursor: pointer;
        }
        .fg input:focus, .fg select:focus { outline: none; border-color: var(--gold); box-shadow: 0 0 0 3px rgba(184,151,59,0.1); }
        .filter-btns { display: flex; gap: 8px; padding-bottom: 1px; }

        /* ── TABLE ── */
        .table-panel { background: var(--white); border: 1px solid var(--border); border-radius: var(--radius); overflow: hidden; }

        .table-head {
            padding: 16px 22px;
            border-bottom: 1px solid var(--border);
            display: flex; align-items: center; justify-content: space-between;
            background: var(--navy);
        }
        .table-head h2 {
            font-family: 'Playfair Display', serif;
            font-size: 1.05rem; font-weight: 400;
            color: var(--white);
            display: flex; align-items: center; gap: 10px;
        }
        .table-head h2 i { color: var(--gold); font-size: 0.9rem; }
        .record-count {
            font-size: 0.72rem; font-weight: 500;
            color: rgba(255,255,255,0.45);
            background: rgba(255,255,255,0.07);
            padding: 4px 12px; border-radius: 100px;
            border: 1px solid rgba(255,255,255,0.1);
        }

        .tbl-wrap { overflow-x: auto; }

        table {
            width: 100%;
            border-collapse: collapse;
        }
        thead th {
            padding: 11px 16px;
            font-size: 0.67rem; font-weight: 700;
            letter-spacing: 0.09em; text-transform: uppercase;
            color: var(--text-light);
            background: var(--cream);
            border-bottom: 1px solid var(--border);
            white-space: nowrap;
            text-align: left;
        }
        tbody td {
            padding: 13px 16px;
            font-size: 0.82rem;
            color: var(--text-dark);
            border-bottom: 1px solid var(--border);
            vertical-align: middle;
        }
        tbody tr:last-child td { border-bottom: none; }
        tbody tr { transition: background .15s; }
        tbody tr:hover { background: rgba(13,27,46,0.02); }

        .cell-id {
            font-size: 0.72rem; font-weight: 700;
            color: var(--text-light);
            font-family: 'DM Sans', monospace;
        }
        .cell-org { font-weight: 600; color: var(--navy); }
        .cell-event { color: var(--text-mid); }
        .cell-rep { font-size: 0.78rem; color: var(--text-light); }
        .cell-date { font-size: 0.76rem; color: var(--text-mid); white-space: nowrap; }
        .rooms-pill {
            display: inline-block;
            font-size: 0.68rem; font-weight: 600;
            background: var(--cream-dark);
            color: var(--text-mid);
            padding: 2px 8px; border-radius: 100px;
        }

        /* Status badges */
        .badge {
            display: inline-flex; align-items: center; gap: 5px;
            padding: 4px 10px; border-radius: 100px;
            font-size: 0.68rem; font-weight: 700;
            letter-spacing: 0.06em; text-transform: uppercase;
            white-space: nowrap;
        }
        .badge::before { content: ''; width: 5px; height: 5px; border-radius: 50%; }
        .badge-pending  { background: var(--warning-bg);  color: var(--warning);  }
        .badge-pending::before  { background: var(--gold); }
        .badge-approved { background: var(--success-bg);  color: var(--success);  }
        .badge-approved::before { background: var(--success); }
        .badge-rejected { background: var(--danger-bg);   color: var(--danger);   }
        .badge-rejected::before { background: var(--danger); }

        /* Action buttons */
        .actions { display: flex; gap: 6px; align-items: center; }
        .act-btn {
            display: inline-flex; align-items: center; gap: 5px;
            padding: 5px 11px;
            border: 1.5px solid transparent;
            border-radius: 5px;
            font-family: 'DM Sans', sans-serif;
            font-size: 0.71rem; font-weight: 600;
            cursor: pointer; transition: all .18s;
            white-space: nowrap;
        }
        .act-btn i { font-size: 0.65rem; }
        .act-view    { border-color: var(--border); background: var(--white); color: var(--text-mid); }
        .act-view:hover { border-color: var(--navy); color: var(--navy); }
        .act-approve { border-color: var(--success-bg); background: var(--success-bg); color: var(--success); }
        .act-approve:hover { background: var(--success); color: white; border-color: var(--success); }
        .act-reject  { border-color: var(--danger-bg); background: var(--danger-bg); color: var(--danger); }
        .act-reject:hover { background: var(--danger); color: white; border-color: var(--danger); }

        /* Empty state */
        .empty-state {
            padding: 72px 20px;
            text-align: center;
            color: var(--text-light);
        }
        .empty-icon {
            width: 64px; height: 64px;
            border-radius: 50%;
            background: var(--cream-dark);
            display: flex; align-items: center; justify-content: center;
            font-size: 1.6rem; color: var(--text-light);
            margin: 0 auto 16px;
        }
        .empty-state h3 {
            font-family: 'Playfair Display', serif;
            font-size: 1.1rem; font-weight: 400;
            color: var(--text-mid); margin-bottom: 6px;
        }
        .empty-state p { font-size: 0.8rem; }

        /* ── MODAL ── */
        .modal-overlay {
            display: none;
            position: fixed; inset: 0;
            background: rgba(10,20,40,0.55);
            z-index: 500;
            align-items: center; justify-content: center;
            backdrop-filter: blur(3px);
        }
        .modal-overlay.open { display: flex; animation: fadeIn .2s ease; }

        @keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }

        .modal-box {
            background: var(--white);
            border-radius: 12px;
            width: 90%; max-width: 620px;
            max-height: 88vh;
            overflow-y: auto;
            box-shadow: 0 32px 80px rgba(10,20,40,0.25);
            animation: slideUp .25s ease;
        }
        @keyframes slideUp {
            from { opacity:0; transform: translateY(20px); }
            to   { opacity:1; transform: translateY(0); }
        }
        .modal-hd {
            background: var(--navy);
            padding: 20px 24px;
            border-radius: 12px 12px 0 0;
            display: flex; align-items: center; justify-content: space-between;
        }
        .modal-hd h3 {
            font-family: 'Playfair Display', serif;
            font-size: 1.1rem; font-weight: 400; color: var(--white);
        }
        .modal-hd .close-btn {
            width: 30px; height: 30px;
            border-radius: 50%;
            background: rgba(255,255,255,0.1);
            border: none; cursor: pointer;
            color: rgba(255,255,255,0.6);
            font-size: 1rem;
            display: flex; align-items: center; justify-content: center;
            transition: all .2s;
        }
        .modal-hd .close-btn:hover { background: rgba(255,255,255,0.2); color: white; }

        .modal-bd {
            padding: 28px 24px;
        }
        .detail-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }
        .detail-item {}
        .detail-item.full { grid-column: 1 / -1; }
        .di-label {
            font-size: 0.65rem; font-weight: 700;
            letter-spacing: 0.09em; text-transform: uppercase;
            color: var(--text-light); margin-bottom: 4px;
        }
        .di-val {
            font-size: 0.85rem; color: var(--text-dark); font-weight: 500;
        }
        .detail-divider {
            grid-column: 1/-1;
            height: 1px; background: var(--border); margin: 4px 0;
        }

        .modal-ft {
            padding: 16px 24px;
            border-top: 1px solid var(--border);
            display: flex; justify-content: flex-end; gap: 10px;
        }

        /* General btn (used in modal footer) */
        .btn {
            display: inline-flex; align-items: center; gap: 7px;
            padding: 10px 20px; border: none; border-radius: var(--radius);
            font-family: 'DM Sans', sans-serif; font-size: 0.8rem; font-weight: 600;
            cursor: pointer; transition: all .2s;
        }
        .btn-ghost { background: transparent; border: 1.5px solid var(--border); color: var(--text-mid); }
        .btn-ghost:hover { border-color: var(--text-mid); background: rgba(0,0,0,0.03); }
        .btn-navy { background: var(--navy); color: white; }
        .btn-navy:hover { background: var(--navy-light); }

        /* Gold divider accent */
        .gold-accent {
            height: 2px;
            background: linear-gradient(90deg, var(--gold) 0%, rgba(184,151,59,0.1) 100%);
            margin-bottom: 28px;
            border-radius: 1px;
        }

        /* ── RESPONSIVE ── */
        @media (max-width: 1100px) {
            .stats-row { grid-template-columns: repeat(2, 1fr); }
            .filter-row { grid-template-columns: 1fr 1fr; }
            .filter-row .filter-btns { grid-column: 1/-1; }
        }
        @media (max-width: 860px) {
            .sidebar { transform: translateX(-100%); transition: transform .3s; }
            .sidebar.open { transform: translateX(0); }
            .main { margin-left: 0; }
            .stats-row { grid-template-columns: 1fr 1fr; }
            .filter-row { grid-template-columns: 1fr; }
            .detail-grid { grid-template-columns: 1fr; }
            .content { padding: 24px 20px; }
            .topbar { padding: 0 20px; }
        }
        @media (max-width: 560px) {
            .stats-row { grid-template-columns: 1fr; }
        }
    </style>
</head>
<body>

<!-- ═══ SIDEBAR ═══ -->
<aside class="sidebar" id="sidebar">
    <div class="sidebar-logo">
        <div class="logo-mark">
            <img src="./cedi_logo.jpg" alt="CCC"
                 onerror="this.style.display='none'; this.parentNode.innerHTML='<i class=\'fas fa-university\'></i>';">
        </div>
        <div class="logo-text">
            <strong>CCC Admin</strong>
            <span>Conference Centre</span>
        </div>
    </div>

    <div class="sidebar-section-label">Main Menu</div>
    <a href="admin_dashboard.php" class="nav-item active">
        <i class="fas fa-th-large"></i> Dashboard
    </a>
    <a href="admin_dashboard.php?status=pending" class="nav-item">
        <i class="fas fa-clock"></i> Pending Requests
        <?php if ($stats['pending'] > 0): ?>
        <span style="margin-left:auto;background:var(--gold);color:var(--navy);font-size:0.62rem;font-weight:700;padding:2px 7px;border-radius:100px;"><?= $stats['pending'] ?></span>
        <?php endif; ?>
    </a>
    <a href="admin_dashboard.php?status=approved" class="nav-item">
        <i class="fas fa-check-circle"></i> Approved
    </a>
    <a href="admin_dashboard.php?status=rejected" class="nav-item">
        <i class="fas fa-times-circle"></i> Rejected
    </a>

    <div class="sidebar-section-label" style="margin-top:8px;">Reports</div>
    <a href="#" class="nav-item" onclick="exportData(); return false;">
        <i class="fas fa-file-export"></i> Export Data
    </a>
    <a href="#" class="nav-item">
        <i class="fas fa-chart-bar"></i> Analytics
    </a>

    <div class="sidebar-section-label" style="margin-top:8px;">Settings</div>
    <a href="#" class="nav-item">
        <i class="fas fa-cog"></i> Settings
    </a>

    <div class="sidebar-footer">
        <a href="admin_logout.php">
            <i class="fas fa-sign-out-alt"></i> Sign Out
        </a>
    </div>
</aside>

<!-- ═══ MAIN ═══ -->
<div class="main">

    <!-- Top bar -->
    <div class="topbar">
        <div class="topbar-left">
            <h1>Booking Dashboard</h1>
            <span>Cedi Conference Centre · University of Ghana</span>
        </div>
        <div class="topbar-right">
            <button class="topbar-btn" onclick="exportData()">
                <i class="fas fa-download"></i> Export
            </button>
            <a href="admin_dashboard.php" class="topbar-btn">
                <i class="fas fa-sync-alt"></i> Refresh
            </a>
            <div class="topbar-avatar">A</div>
        </div>
    </div>

    <!-- Content -->
    <div class="content">

        <!-- Gold accent line -->
        <div class="gold-accent"></div>

        <!-- Stats -->
        <div class="stats-row">
            <div class="stat-card s-total">
                <div class="stat-top">
                    <span class="stat-label">Total Bookings</span>
                    <div class="stat-icon-wrap"><i class="fas fa-calendar-check"></i></div>
                </div>
                <div class="stat-number"><?= $stats['total'] ?></div>
                <div class="stat-sub">All time submissions</div>
            </div>
            <div class="stat-card s-pending">
                <div class="stat-top">
                    <span class="stat-label">Awaiting Review</span>
                    <div class="stat-icon-wrap"><i class="fas fa-hourglass-half"></i></div>
                </div>
                <div class="stat-number"><?= $stats['pending'] ?></div>
                <div class="stat-sub">Requires action</div>
            </div>
            <div class="stat-card s-approved">
                <div class="stat-top">
                    <span class="stat-label">Approved</span>
                    <div class="stat-icon-wrap"><i class="fas fa-check-circle"></i></div>
                </div>
                <div class="stat-number"><?= $stats['approved'] ?></div>
                <div class="stat-sub">Confirmed bookings</div>
            </div>
            <div class="stat-card s-rejected">
                <div class="stat-top">
                    <span class="stat-label">Rejected</span>
                    <div class="stat-icon-wrap"><i class="fas fa-times-circle"></i></div>
                </div>
                <div class="stat-number"><?= $stats['rejected'] ?></div>
                <div class="stat-sub">Declined requests</div>
            </div>
        </div>

        <!-- Filters -->
        <div class="panel">
            <div class="panel-head">
                <h2><i class="fas fa-sliders-h"></i> Filter Bookings</h2>
            </div>
            <div class="panel-body">
                <form method="GET" action="">
                    <div class="filter-row">
                        <div class="fg">
                            <label>Status</label>
                            <select name="status">
                                <option value="all"     <?= $status_filter==='all'      ? 'selected':'' ?>>All Statuses</option>
                                <option value="pending" <?= $status_filter==='pending'  ? 'selected':'' ?>>Pending</option>
                                <option value="approved"<?= $status_filter==='approved' ? 'selected':'' ?>>Approved</option>
                                <option value="rejected"<?= $status_filter==='rejected' ? 'selected':'' ?>>Rejected</option>
                            </select>
                        </div>
                        <div class="fg">
                            <label>Search</label>
                            <input type="text" name="search" placeholder="Organisation, event, representative…" value="<?= htmlspecialchars($search_term) ?>">
                        </div>
                        <div class="fg">
                            <label>From Date</label>
                            <input type="date" name="date_from" value="<?= $date_from ?>">
                        </div>
                        <div class="fg">
                            <label>To Date</label>
                            <input type="date" name="date_to" value="<?= $date_to ?>">
                        </div>
                        <div class="filter-btns">
                            <button type="submit" class="btn btn-navy" style="padding:9px 18px;font-size:0.78rem;">
                                <i class="fas fa-filter"></i> Apply
                            </button>
                            <a href="admin_dashboard.php" class="btn btn-ghost" style="padding:9px 14px;font-size:0.78rem;">
                                <i class="fas fa-redo"></i>
                            </a>
                        </div>
                    </div>
                </form>
            </div>
        </div>

        <!-- Table -->
        <div class="table-panel">
            <div class="table-head">
                <h2><i class="fas fa-list-ul"></i> Booking Requests</h2>
                <span class="record-count"><?= count($bookings) ?> record<?= count($bookings)!==1?'s':'' ?></span>
            </div>
            <div class="tbl-wrap">
                <?php if (count($bookings) > 0): ?>
                <table>
                    <thead>
                        <tr>
                            <th>Ref</th>
                            <th>Organisation</th>
                            <th>Event</th>
                            <th>Representative</th>
                            <th>Event Dates</th>
                            <th>Rooms</th>
                            <th>Status</th>
                            <th>Submitted</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($bookings as $b): ?>
                        <tr>
                            <td><span class="cell-id">#<?= str_pad($b['id'], 4, '0', STR_PAD_LEFT) ?></span></td>
                            <td><span class="cell-org"><?= htmlspecialchars($b['organization_name']) ?></span></td>
                            <td><span class="cell-event"><?= htmlspecialchars($b['event_name']) ?></span></td>
                            <td><span class="cell-rep"><?= htmlspecialchars($b['representative_name']) ?></span></td>
                            <td><span class="cell-date"><?= htmlspecialchars($b['event_dates']) ?></span></td>
                            <td>
                                <?php
                                $rooms = array_filter(explode(',', $b['rooms'] ?? ''));
                                $rc    = count($rooms);
                                echo $rc > 2
                                    ? "<span class='rooms-pill'>{$rc} rooms</span>"
                                    : "<span class='rooms-pill'>" . htmlspecialchars($b['rooms'] ?? '—') . "</span>";
                                ?>
                            </td>
                            <td>
                                <span class="badge badge-<?= $b['status'] ?>">
                                    <?= ucfirst($b['status']) ?>
                                </span>
                            </td>
                            <td><span class="cell-date"><?= date('d M Y', strtotime($b['submission_date'])) ?></span></td>
                            <td>
                                <div class="actions">
                                    <button class="act-btn act-view" onclick="viewBooking(<?= $b['id'] ?>)">
                                        <i class="fas fa-eye"></i> View
                                    </button>
                                    <?php if ($b['status'] === 'pending'): ?>
                                    <button class="act-btn act-approve" onclick="updateStatus(<?= $b['id'] ?>, 'approved')">
                                        <i class="fas fa-check"></i> Approve
                                    </button>
                                    <button class="act-btn act-reject" onclick="updateStatus(<?= $b['id'] ?>, 'rejected')">
                                        <i class="fas fa-times"></i> Reject
                                    </button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <?php else: ?>
                <div class="empty-state">
                    <div class="empty-icon"><i class="fas fa-inbox"></i></div>
                    <h3>No bookings found</h3>
                    <p>Try adjusting your filters or check back later for new requests.</p>
                </div>
                <?php endif; ?>
            </div>
        </div>

    </div><!-- /content -->
</div><!-- /main -->

<!-- ═══ MODAL ═══ -->
<div class="modal-overlay" id="modalOverlay">
    <div class="modal-box">
        <div class="modal-hd">
            <h3>Booking Details</h3>
            <button class="close-btn" onclick="closeModal()"><i class="fas fa-times"></i></button>
        </div>
        <div class="modal-bd" id="modalBody">
            <div style="text-align:center;padding:40px;color:var(--text-light);">
                <i class="fas fa-spinner fa-spin fa-2x"></i>
            </div>
        </div>
        <div class="modal-ft">
            <button class="btn btn-ghost" onclick="closeModal()">Close</button>
        </div>
    </div>
</div>

<script>
    // Modal
    const overlay = document.getElementById('modalOverlay');

    function viewBooking(id) {
        overlay.classList.add('open');
        document.getElementById('modalBody').innerHTML = '<div style="text-align:center;padding:40px;color:var(--text-light);"><i class="fas fa-spinner fa-spin fa-2x"></i></div>';
        fetch(`get_booking_details.php?id=${id}`)
            .then(r => r.text())
            .then(html => { document.getElementById('modalBody').innerHTML = html; })
            .catch(() => { document.getElementById('modalBody').innerHTML = '<p style="color:var(--danger);padding:20px;">Error loading booking details.</p>'; });
    }

    function closeModal() { overlay.classList.remove('open'); }
    overlay.addEventListener('click', e => { if (e.target === overlay) closeModal(); });
    document.addEventListener('keydown', e => { if (e.key === 'Escape') closeModal(); });

    // Status update
    function updateStatus(id, status) {
        const label = status === 'approved' ? 'approve' : 'reject';
        if (!confirm(`Are you sure you want to ${label} booking #${String(id).padStart(4,'0')}?`)) return;
        fetch('update_booking_status.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: `id=${id}&status=${status}`
        })
        .then(r => r.json())
        .then(d => {
            if (d.success) location.reload();
            else alert('Error: ' + d.message);
        })
        .catch(() => alert('Network error. Please try again.'));
    }

    // Export
    function exportData() {
        const params = new URLSearchParams(window.location.search);
        window.open(`export_bookings.php?${params.toString()}`, '_blank');
    }
</script>

</body>
</html>